#!/usr/bin/env python3
# I25 — Tie-Kernel Primitivity & RNG Determinism
# CONTROL: present-act, boolean/ordinal; decisions are deterministic unless an exact tie occurs.
# RNG is called ONLY when a tie occurs; otherwise selection is deterministic (content order).
# TIE KERNEL: Build a strictly positive (primitive) row-stochastic matrix M from a binary template + η.
# Compute PF vector v via power iteration; Born weights p_i ∝ v_i^α (α≥1, default 2).
# TESTS:
#  • Primitivity: M>0 (all entries positive) and v>0.
#  • Isolation: RNG calls == #ties; RNG not called on non-ties.
#  • Determinism: same seed ⇒ identical sequence; different seed ⇒ different sequence.
#  • Born match: empirical tie frequencies match p within L1 tolerance.
# OUTPUTS: metrics/*.csv, audits/i25_audit.json, run_info/result_line.txt

import argparse, csv, json, math, os, random, sys
from typing import List, Dict, Tuple
from datetime import datetime, timezone

def utc_timestamp() -> str:
    return datetime.now(timezone.utc).strftime("%Y-%m-%dT%H-%M-%SZ")

def ensure_dirs(root: str, subs: List[str]):
    for s in subs:
        os.makedirs(os.path.join(root, s), exist_ok=True)

def write_text(path: str, text: str):
    with open(path, "w", encoding="utf-8") as f: f.write(text)

def dump_json(path: str, obj: dict):
    with open(path, "w", encoding="utf-8") as f: json.dump(obj, f, indent=2, sort_keys=True)

# ---------- PF via power iteration ----------
def power_iteration(M: List[List[float]], tol: float=1e-12, itmax: int=10000) -> List[float]:
    n = len(M)
    v = [1.0/n]*n
    for _ in range(itmax):
        w = [0.0]*n
        for i in range(n):
            row = M[i]
            s = 0.0
            # right eigenvector of M^T (equivalently left of M); we can multiply by M^T:
            for j in range(n):
                s += M[j][i]*v[j]
            w[i] = s
        ssum = sum(w)
        if ssum == 0.0:
            break
        w = [x/ssum for x in w]
        diff = sum(abs(w[i]-v[i]) for i in range(n))
        v = w
        if diff < tol:
            break
    return v

def l1(a: List[float], b: List[float]) -> float:
    return sum(abs(x-y) for x,y in zip(a,b))

def sample_from_weights(rng: random.Random, weights: List[float], labels: List[str]) -> str:
    # cumulative
    s = 0.0
    u = rng.random()
    for w,lbl in zip(weights, labels):
        s += w
        if u <= s:
            return lbl
    return labels[-1]

# ---------- deterministic present-act control with ties ----------
def run_sequence(params: dict, seed: int) -> Dict[str, object]:
    labels = params["labels"]                     # e.g., ["A","B","C","D"]
    N = len(labels)
    H = int(params["H"])                          # total ticks
    tie_every = int(params["tie_every"])          # ties every T ticks
    # deterministic non-tie rule: choose labels[t % N] (content order)
    # tie kernel & weights:
    eta = float(params["eta"])
    born_pow = float(params["born_power"])
    # binary template adjacency (dense off-diagonal + diag)
    A = [[1 if True else 0 for _ in range(N)] for _ in range(N)]
    # add η everywhere and row-normalize to get primitive stochastic M
    M = []
    for i in range(N):
        row = []
        for j in range(N):
            row.append(A[i][j] + eta)
        s = sum(row)
        row = [x/s for x in row]
        M.append(row)
    # Primitivity check: all entries strictly > 0
    primitive = all(M[i][j] > 0.0 for i in range(N) for j in range(N))
    # PF vector and Born weights
    v = power_iteration(M, tol=1e-12, itmax=10000)
    vpos = all(x > 0.0 for x in v)
    # normalize v
    vs = sum(v)
    v = [x/vs for x in v]
    p = [max(0.0, x**born_pow) for x in v]
    ps = sum(p)
    p = [x/ps for x in p]

    rng = random.Random(seed)
    seq = []
    tie_mask = []
    rng_calls = 0

    # run ticks
    for t in range(H):
        is_tie = (t % tie_every == 0)  # deterministic schedule of ties
        tie_mask.append(1 if is_tie else 0)
        if is_tie:
            # RNG ONLY at ties
            choice = sample_from_weights(rng, p, labels)
            rng_calls += 1
        else:
            # deterministic pick (content order)
            choice = labels[t % N]
        seq.append(choice)

    # tallies over tie events
    tie_total = sum(tie_mask)
    tie_counts = {lbl:0 for lbl in labels}
    for s, tm in zip(seq, tie_mask):
        if tm == 1:
            tie_counts[s] += 1
    tie_freq = [tie_counts[lbl]/tie_total if tie_total>0 else 0.0 for lbl in labels]

    return {
        "M": M, "v": v, "p": p,
        "primitive": primitive, "vpos": vpos,
        "labels": labels, "H": H, "tie_every": tie_every,
        "rng_calls": rng_calls, "tie_events": tie_total,
        "seq": seq, "tie_mask": tie_mask, "tie_counts": tie_counts, "tie_freq": tie_freq
    }

def run_i25(manifest: dict, outdir: str) -> Dict[str, object]:
    labels = manifest["labels"]
    # 1) run with seed0
    run0 = run_sequence(manifest, int(manifest["seed0"]))
    # 2) replay with same seed0 → must be identical
    run0b = run_sequence(manifest, int(manifest["seed0"]))
    # 3) run with different seed1 → must differ (sequence) but match frequencies to p within tol
    run1 = run_sequence(manifest, int(manifest["seed1"]))

    # checks
    primitive_ok = bool(run0["primitive"] and run0["vpos"])
    rng_isolation_ok = bool(run0["rng_calls"] == run0["tie_events"])
    same_seed_ok = (run0["seq"] == run0b["seq"])
    diff_seed_differs = (run0["seq"] != run1["seq"])

    # frequency checks: L1 error ≤ tol_l1 for both seeds
    tol_l1 = float(manifest["acceptance"]["l1_tol"])
    l1_0 = l1(run0["tie_freq"], run0["p"])
    l1_1 = l1(run1["tie_freq"], run1["p"])
    born_match_ok = bool(l1_0 <= tol_l1 and l1_1 <= tol_l1)

    # same-seed reproducibility implies also tie_counts equal
    tie_counts_equal = all(run0["tie_counts"][lbl] == run0b["tie_counts"][lbl] for lbl in labels)

    # Build audit
    audit = {
        "sim": "I25_kernel_rng",
        "labels": labels,
        "primitive_ok": primitive_ok,
        "rng_isolation_ok": rng_isolation_ok,
        "same_seed_ok": same_seed_ok,
        "tie_counts_equal": tie_counts_equal,
        "diff_seed_differs": diff_seed_differs,
        "l1_error_seed0": l1_0,
        "l1_error_seed1": l1_1,
        "p_pf_born": run0["p"],
        "tie_freq_seed0": run0["tie_freq"],
        "tie_freq_seed1": run1["tie_freq"],
        "rng_calls": run0["rng_calls"],
        "tie_events": run0["tie_events"],
        "accept": manifest["acceptance"]
    }

    passed = bool(
        primitive_ok and rng_isolation_ok and same_seed_ok and tie_counts_equal and diff_seed_differs and born_match_ok
    )
    audit["passed"] = passed

    # write metrics
    with open(os.path.join(outdir,"outputs/metrics","i25_tie_freqs.csv"), "w", newline="", encoding="utf-8") as f:
        w = csv.writer(f); w.writerow(["label","p_theory","freq_seed0","freq_seed1"])
        for i,lbl in enumerate(labels):
            w.writerow([lbl, f"{run0['p'][i]:.6f}", f"{run0['tie_freq'][i]:.6f}", f"{run1['tie_freq'][i]:.6f}"])
    with open(os.path.join(outdir,"outputs/metrics","i25_kernel.csv"), "w", newline="", encoding="utf-8") as f:
        w = csv.writer(f); w.writerow(["row","col","M_ij"])
        for i,row in enumerate(run0["M"]):
            for j,val in enumerate(row):
                w.writerow([i,j,f"{val:.9f}"])

    # persist audit
    dump_json(os.path.join(outdir,"outputs/audits","i25_audit.json"), audit)

    # result line
    line = (
        "I25 PASS={p} primitive={pr} rng_isolated={ri} same_seed={ss} "
        "diff_seed={ds} L1_0={l0:.4f} L1_1={l1:.4f}"
        .format(p=passed, pr=primitive_ok, ri=rng_isolation_ok, ss=same_seed_ok,
                ds=diff_seed_differs, l0=l1_0, l1=l1_1)
    )
    write_text(os.path.join(outdir,"outputs/run_info","result_line.txt"), line)
    print(line)
    return audit

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--manifest", required=True)
    ap.add_argument("--outdir", required=True)
    args = ap.parse_args()

    ensure_dirs(args.outdir, "config","outputs/metrics","outputs/audits","outputs/run_info","logs")
    with open(args.manifest,"r",encoding="utf-8") as f:
        M = json.load(f)
    # persist manifest and env
    dump_json(os.path.join(args.outdir,"config","manifest_i25.json"), M)
    write_text(os.path.join(args.outdir,"logs","env.txt"),
               "utc={}\nos={}\npython={}\n".format(utc_timestamp(), os.name, sys.version.split()[0]))

    run_i25(M, args.outdir)

if __name__ == "__main__":
    main()
